function [min_ht,min_wid]= gui_position(layout,param,do_not_position_flag)
% This function should position uicontrols, uipanels, axes, etc, within a
% the given container object. This function will be used to do an initial
% layout, and also any re-layout after resizing. 


%% Process and Validate inputs
if nargin<3
    do_not_position_flag = false;
end
[layout,parent,leftmargin,rightmargin,topmargin,botmargin] = process_header(layout,param);

%% Some basic spacing variables
interline = param.spacing.min_interline;

%% Find container size
pos = get(parent,'Position');
container_width = pos(3); %maybe should subtract any border here, maybe could be rolled into margins later
container_height = pos(4);



%% Set up element classification code list
% the command list is
com_list = {'settab','tab','space','vspace',...
    'hrubber','vrubber','left','right','center',...
    'vtopalign','vbotalign','vmidalign','vbasealign',...
    'ghost','ignoreline'};
%%%% IMPORTANT NOTE: All commands must be uniquely identifiable by their
%%%% first three letters!!!  This is so that there can be variable end
%%%% portions (e.g. numbers on the spaces or tabs, etc).
class_key = {com_list{:},'handle','sublayout','badcom','unknown'};

findcode([],class_key); % initialize findcode with the class key list

%% Build matrices
[h,w,spaceafter,classcode,baselines,xstretch,ystretch] = build_matrices(layout,param,class_key);

%% Make a list of non-ignoreline rows
[ignorelines, dum] = find(classcode==findcode('ignoreline'));
real_lines = 1:length(layout);
real_lines(ignorelines) = [];

%% Determine minimum layout height and width
row_widths = sum(w,2)+sum(spaceafter,2)+leftmargin+rightmargin;  %these are minimal row widths ignoring all tabs and 
min_wid = max(row_widths(real_lines));

natural_h = h + baselines;
row_heights = max(natural_h,[],2);
num_interlines = length(real_lines)-1;
min_ht = sum(row_heights(real_lines))+interline*num_interlines+topmargin+botmargin;
if do_not_position_flag
    % If this flag is set, then constructing min_wid and min_ht was the
    % whole point of the call, and we can just return them
    return
end
%% Resize container if too small

curpos = get(parent,'Position'); %what if units not pixels?
if container_width<min_wid || container_height<min_ht
    warning('LogiGUI:ContainerTooSmall','The container size is smaller than the minimal size of it''s contents, it is being resized to be at least the minimal size');
    % Resize container (grows to the right and down, leaving upper left
    % corner unchanged)
    container_width = max(min_wid,curpos(3)); % new container width
    container_height = max(min_ht,curpos(4)); % new container height
    set(parent,'Position',[curpos(1),curpos(2)-(container_height-curpos(4)),container_width,container_height]);
    %set(parent,'Position',[curpos(1:2),container_width,container_height]); 
    
end

    
%% Process rubber lengths and interline spacing
% Horizontal first
% Determine if there's any extra space
extra_hspace = container_width-row_widths;
% Horizontal rubber spaces and other h-stretchy objects
% on each row, add up any hrubber weights and divide extra space according
% to weights
xrubber_totals = sum(xstretch,2); 
warning('off','MATLAB:divideByZero');
hspaces_per_hrubber_unit = extra_hspace./xrubber_totals;
warning('on','MATLAB:divideByZero');
hspaces_per_hrubber_unit(isnan(hspaces_per_hrubber_unit) | isinf(hspaces_per_hrubber_unit)) = 0;
xrubberspace = zeros(size(xstretch));
for i = 1:size(xstretch,1)
    xrubberspace(i,:) = xstretch(i,:)*hspaces_per_hrubber_unit(i);
end

% Vertical
% Determine natural row heights using h and baselines
% Check for extra space, don't forget to drop ignorelines from calcs
extra_vspace = container_height-sum(row_heights(real_lines))-interline*num_interlines-topmargin-botmargin;

% Set vertical rubber spaces or stretch interline spacing
ystretch_maxs = max(ystretch,[],2);
if any(ystretch_maxs(real_lines))
    %stretch_interline = false;
    vspace_per_vrubber_unit = extra_vspace/sum(ystretch_maxs(real_lines));
    row_heights = row_heights+ystretch_maxs*vspace_per_vrubber_unit;
else
    %stretch_interline = true;
    % Distribute extra space among the top and bottom margins and the
    % interline spacing
    addspace = extra_vspace/(num_interlines+2);
    interline = interline+addspace;
    topmargin = topmargin+addspace;
    botmargin = botmargin+addspace;
end

%% Define the bottom and top of rows
% Find row boundaries based on row heights and interline spacing
% There's a question about what to do about ignorelines, they're
% present in row_heights, but they don't need to be positioned
% relative to the container because their only processing function is
% to set tabs.  For now, we'll just give them zero for Rbot and zero for
% Rtop, i.e. skip them during processing

% Initialize Rbots and Rtops
Rbots = zeros(size(row_heights));
Rtops = Rbots;

% Start from the top and work down
current_position = container_height-topmargin;
for i = 1:length(row_heights)
    if ismember(i,real_lines)
        Rtops(i) = current_position;
        current_position = current_position - row_heights(i);
        Rbots(i) = current_position;
        current_position = current_position-interline;
    end
end
% Correct final current position to reflect bottom margin rather
% than interline spacing
current_position = current_position+interline-botmargin;
        


%% Start positioning!

% Loop over layout rows
for i = 1:length(layout)
    % Check if ignoreline
    if ismember(i,ignorelines)
        ignoreline = true;
    else
        ignoreline = false;
    end
    currow = layout{i};
    % Set line top, line bottom, and baseline
    Rtop = Rtops(i);
    Rbot = Rbots(i);
    % Set default vertical alignment
    valign = 'base';
    % Classify the row as hrubber-governed or alignment-governed
    % if alignment-governed, set default left alignment
    if any(xstretch(i,:))
        % Row is hrubber-governed
        hrubber_row = true;
        % Break row into chunks at tabs
        [startpoints,finishpoints] = find_chunks(currow,classcode(i,:),'tab');
    else
        % Row is alignment-governed
        hrubber_row = false;
        % Break row into chunks at tabs and horizontal alignment changes
        [startpoints,finishpoints] = find_chunks(currow,classcode(i,:),'tab','left','right','center');
    end
    
    % Loop over chunks
    for startind = 1:length(startpoints)
        start = startpoints(startind);
        finish = finishpoints(startind);
        curleft = leftmargin; % start the left margin
        isghost = false; % can't ghost across chunk boundaries
        chunk_width = sum(w(i,start:finish))+sum(spaceafter(i,start:finish))+sum(xrubberspace(i,start:finish));
        %htoalign = []; % list of handles to align
            
        % Loop over elements within chunks, positioning them
        for curcol = start:finish
            element = currow{curcol}; % current element
            curclass = class_key{classcode(i,curcol)};
            switch curclass
                case {'left','center','right'}
                    if ~hrubber_row
                        switch curclass
                            case 'left'
                                %don't do anything, already set up
                            case 'center'
                                curleft = (container_width-leftmargin-rightmargin-chunk_width)/2+leftmargin;
                            case 'right'
                                curleft = container_width-chunk_width-rightmargin;
                        end
                    else  % hrubber aligned, so command ignored
                        warning('LogiGUI:AlignmentCommandIgnored',['A horizontal alignment '...
                            'command was found on a row which has hrubber elements '...
                            'or a stretchy container. These are incompatible, so the '...
                            'horizontal alignment command is being ignored']);
                    end
                case 'settab'
                    % Set tab value to current position
                    tabname = element(4:end);
                    tabs.(tabname) = curleft;
                case 'tab'
                    % a tab sets curleft directly
                    try
                        curleft = tabs.(element);
                    catch
                        error('LogiGUI:UnSetTabError',['You have referenced a tab you didn''t set (called ''' element ''')']);
                    end
                case 'space'
                    % Add the space from w
                    curleft = curleft + w(i,curcol);
                case 'hrubber'
                    % Add the space from xrubberspace
                    curleft = curleft + xrubberspace(i,curcol);
                case 'vtopalign'
                    valign = 'top';
                case 'vbasealign'
                    valign = 'base';
                case 'vmidalign'
                    valign = 'mid';
                case 'vbotalign'
                    valign = 'bot';
                case 'handle'
                    % Position this element, respecting it's vertical
                    % alignment and ghosting state, and assigning it's
                    % parent
                    if ~isghost && ~ignoreline
                        % Set parent
                        set(element,'Parent',parent);
                        % Determine left, bottom, width, and height
                        elem_l = curleft;
                        elem_w = w(i,curcol);
                        elem_h = h(i,curcol);
                        switch valign
                            case 'base'
                                % Bottom of element sits on baseline
                                elem_b = Rbot+baselines(i,curcol);
                            case 'bot'
                                elem_b = Rbot;
                            case 'mid'
                                elem_b = Rbot+(Rtop-Rbot-elem_h)/2;
                            case 'top'
                                elem_b = Rtop-elem_h;
                            otherwise
                                error('LogiGUI:BadVerticalAlignValue',['Vertical alignment ' valign ' not understood']);
                        end
                        % Checkboxes look bad on the baseline, they need to
                        % be lifted...
                        if strcmp(get(element,'Type'),'uicontrol') && strcmp(get(element,'Style'),'checkbox')
                            elem_b = elem_b+param.spacing.checkboxlift;
                        end
                        % Position element
                        set_pos(element,[elem_l,elem_b,elem_w,elem_h]);
                    end
                    
                    % Add to curleft even if ghost or ignorline element
                    curleft = curleft+w(i,curcol);              
                    
                case 'sublayout'
                    % Get subcontainer handle
                    [dum,subcontainer_h] = process_header(element,param);
                    % Parent subcontainer
                    set(subcontainer_h,'Parent',parent);
                    % Determine subcontainer size
                    if ystretch(i,curcol)
                        elem_h = Rtop-Rbot;
                        elem_b = Rbot;
                    else
                        elem_h = h(i,curcol);
                        switch valign
                            case 'base'
                                elem_b = Rbot+baselines(i,curcol);
                            case 'bot'
                                elem_b = Rbot;
                            case 'mid'
                                elem_b = Rbot+(Rtop-Rbot-h(i,curcol))/2;
                            case 'top'
                                elem_b = Rtop-h(i,curcol);
                        end
                    end
                    
                    if ~isghost && ~ignoreline
                        % Position subcontainer
                        if xstretch(i,curcol)
                            % If x-strtechy, then fits contents + rubber
                            % space
                            set_pos(subcontainer_h,[curleft,elem_b,w(i,curcol)+xrubberspace(i,curcol),elem_h]);
                        else
                            % If non-stretchy, then stays the same size,
                            % unless contents don't fit, in which case it
                            % minimally resizes until they do fit.
                            curpos = get(subcontainer_h,'Position');
                            elem_w = max(curpos(3),w(i,curcol));
                            %elem_h = max(curpos(4),h(i,curcol)); % height already taken care of above
                            set_pos(subcontainer_h,[curleft,elem_b,elem_w,elem_h]);
                        end
                        % Nest gui_position calls
                        gui_position(element,param);
                    end
                    
                    % Move curleft even if ghost or ignoreline
                    curleft = curleft+w(i,curcol)+xrubberspace(i,curcol);
                    
                case 'ghost'
                    %Toggle ghosting state
                    isghost = ~isghost;
                    
                case {'badcom','unknown','vrubber','vspace','ignoreline'}
                    % ignore these commands
                otherwise
                    error('LogiGUI:UnhandledCommand',['Commands of class ' curclass ' are not handled at all when positioning elements!']);        
            end
            
            % Add space after current element according to spaceafter matrix
            curleft = curleft+spaceafter(i,curcol);
            
        end % of current element
    end % of current chunk
end % of layout row


end % of gui_position function



%% %%%%%%%%%%%%% SUBFUNCTIONS %%%%%%%%%%%%%%%%%

%% process_header
function [layout,parent,leftmargin,rightmargin,topmargin,botmargin,xstretch,ystretch] = process_header(layout,param)
% looks through the header line of a layout and sets variables

% Set up defaults
parent = gcf;
leftmargin = param.spacing.parent.leftmargin;
rightmargin = param.spacing.parent.rightmargin;
topmargin = param.spacing.parent.topmargin;
botmargin = param.spacing.parent.botmargin;
xstretch = param.spacing.parent.xstretch;
ystretch = param.spacing.parent.ystretch;

% A parent tag defines whether the first line is a header or not
if any(strcmpi('parent',layout{1}))
    header = layout{1};
    layout = layout(2:end);
else
    return
end    

% Set up for parse_inputs
varargin = header;
arg_list = {'leftmargin','rightmargin','topmargin','botmargin','parent','xstretchfactor','ystretchfactor'};
variables = {'leftmargin','rightmargin','topmargin','botmargin','parent','xstretch','ystretch'};

parse_inputs;

end % of process_header function

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% findcode
function code = findcode(element,init_classcode_key)
% this function takes a single command name or a series of
% command names and returns a list of corresponding codes for those
% commands.
% [tabcode, rightcode] = findcode(classcode_key,'tab','right');
persistent classcode_key
if nargin>1
    classcode_key = init_classcode_key;
end

if isempty(classcode_key)
    error('LogiGUI:findcodeNotInitialized','The findcode function must be initialized with the classcode_key before being used to find class codes');
end

code = find(strncmpi(element,classcode_key,3));
if isempty(code)
    code = 0;
end

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% getnum
function num = getnum(elem)
%find the first colon, and then eval everything after that
ind = find(elem==':',1,'first');
num = eval(elem(ind+1:end));
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% get_hw
function [hout,wout] = get_hw(element)
oldunits = get(element,'Units');
set(element,'Units','pixels');

if strcmp(get(element,'Type'),'axes')
    posprop = get(element,'ActivePositionProperty');
else
    posprop = 'Position';
end
pos = get(element,posprop);
hout = pos(4);
wout = pos(3);
set(element,'Units',oldunits);

% Check Element for AppData which would have minimum heights and widths
% which should override the item's actual size...
% (this is mostly used to allow axes to shrink again...)
if isappdata(element,'min_ht')
    hout = getappdata(element,'min_ht');
end
if isappdata(element,'min_w')
    wout = getappdata(element,'min_w');
end

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% set_pos
function set_pos(element,pos)

oldunits = get(element,'Units');

%%%% TOTAL KLUGE %%%%
% maximizes all axes within their containers, fix once stretchy objects
% specified through parameter styling has been implemented
if strcmp(get(element,'Type'),'axes')
    set(element,'Units','pixels');
    labels_offset = get(element,'TightInset');
    parent = get(element,'Parent');
    oldunits_parent = get(parent,'Units');
    set(parent,'Units','pixels');
    parent_pos = get(parent,'Position');
    parent_size = parent_pos(3:4);
    set(element, 'Position', [labels_offset(1:2),(parent_size-4-(labels_offset(3:4)+labels_offset(1:2)))]);
    %set(element,'Units','normalized');
    %set(element,'OuterPosition',[.05,.05,.9,.9]);
    set(element,'Units',oldunits);
    set(parent,'Units',oldunits_parent);
    return
end
%%%% END TOTAL KLUGE %%%%

set(element,'Units','pixels');
if strcmp(get(element,'Type'),'axes')
    posprop = get(element,'ActivePositionProperty');
else
    posprop = 'Position';
end
set(element,posprop,pos);
set(element,'Units',oldunits);

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% find_chunks
function [startpoints,finishpoints] = find_chunks(currow,varargin)
start = zeros(size(currow));
start(1) = 1;
for i = 1:length(varargin)
    temp = strncmpi(varargin{i},currow,3);
    start = start | temp;
end
startpoints = find(start);
finishpoints = [startpoints(2:end)-1,length(currow)];
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% build_matrices
function [h,w,spaceafter,classcode,baselines,xstretch,ystretch] = build_matrices(layout,param,class_key)

% Initialize matrices
for i = 1:length(layout)
    num_entries(i) = length(layout{i}); %#ok<AGROW>
end
max_num_entries = max(num_entries);
numrows = length(layout);
m = max_num_entries;
% Initialize matrices with zeros
w = zeros(numrows,m);  % basic element widths
h = w; % basic element heights
classcode = w; % element classification
vshift = w; % vertical shifts (to keep track of baselines)
xstretch = w; % x stretch factors for elements (only containers)
ystretch = w; % y stretch factors for elements (rows)
spaceafter = w; % keep track of interelement spacings 

% loop over each element in each row, and build w, h, classcode, vshift,
% xstretch, ystretch, and spaceafter matrices
for i = 1:length(layout)
    for j = 1:length(layout{i})
        element = layout{i}{j};
        % Is it a handle?
        if ishandle(element)
            classcode(i,j) = findcode('handle');
            [h(i,j),w(i,j)] = get_hw(element); % get height and width
            spaceafter(i,j) = follow_space(element,param);
            continue
        end
        % Is it a sublayout?
        if iscell(element)
            classcode(i,j) = findcode('sublayout');
            % get minimal height and width of sublayout
            [min_h,min_w] = gui_position(element,param,1);
            
            % Process sublayout header to get handle and stretchyness
            % values
            [dum,subcontainer,dum,dum,dum,dum,xstretch(i,j),ystretch(i,j)] = process_header(element,param);
            % Set basic width and height
            [cur_h,cur_w] = get_hw(subcontainer);
            if xstretch(i,j)
                % if stretchy, then the basic width should be the minimum width
                w(i,j) = min_w;
            else
                % if not stretchy, then the basic width should be the
                % existing width, unless that's too small, in which case it
                % should be resized to the minimum width
                w(i,j) = max(min_w,cur_w);                
            end
            if ystretch(i,j)
                h(i,j) = min_h;
            else
                h(i,j) = max(min_h,cur_h);
            end
            
            % Set space after
            spaceafter(i,j) = follow_space(subcontainer,param); % base this on container line of sublayout? 
            continue
        end
        % Is it a string?
        if ischar(element)
            % Is it a command?
            if findcode(element)
                classcode(i,j) = findcode(element);
                switch class_key{findcode(element)}
                    case 'vspace'
                        vshift(i,j) = getnum(element);
                    case 'hrubber'
                        xstretch(i,j) = getnum(element);
                    case 'vrubber'
                        ystretch(i,j) = getnum(element);
                    case 'space'
                        w(i,j) = getnum(element);
                        
                end
            else % not a command, but it is a string, probably a typo
                classcode(i,j) = findcode('badcom');
                warning('LogiGUI:BadCommand',['Command ''', element,''' not understood in layout']);
            end
        else
            %not a handle, not a sublayout, and not a string... looks like
            %an error
            classcode(i,h) = findcode('unknown');
            warning('LogiGUI:UnknownLayoutElement','The following element has been encountered in a layout and does not appear to be an object handle, a command, or a sublayout:');
            disp(element);
        end
        
    end
end

% Construct baseline matrix
% Assume the baseline starts at 0 until that's contradicted
% Then baseline is 0 before first vshift.
baselines = zeros(size(vshift));
for rownum = 1:size(vshift,1)
    vshiftinds = find(vshift(rownum,:));
    for i = 1:length(vshiftinds)
        ind = vshiftinds(i);
        baselines(rownum,ind:end) = baselines(rownum,ind:end)+vshift(rownum,ind);
    end
end
% Now, set lowest baseline to zero
for i = 1:size(baselines,1)
    baselines(i,:) = baselines(i,:)-min(baselines(i,:));
end

% Reset baselines to zero for any element with a vertical alignment
% other than 'base'
nonbasealign = classcode==findcode('vtopalign')|classcode==findcode('vmidalign')|classcode==findcode('vbotalign');
basealign = classcode==findcode('vbasealign');
% Go row by row
for rownum = 1:size(baselines,1)
    dropbaseline = false;
    for colnum = 1:size(baselines,2)
        if nonbasealign(rownum,colnum)
            dropbaseline = true;
        elseif basealign(rownum,colnum)
            dropbaseline = false;
        end
        if dropbaseline
            baselines(rownum,colnum) = 0;
        end
    end
end


end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% follow_space
function space = follow_space(obj_handle,param)
% Returns the appropriate amount of space to stick after the input
% object

stylegroup = getappdata(obj_handle,'StyleGroup');

%Find object type
switch get(obj_handle,'Type')
    case 'uicontrol'
        if ~isempty(stylegroup) && isfield(param.(stylegroup),'afterspace') && ~isempty(param.(stylegroup).afterspace)
            space = param.(stylegroup).afterspace;
        elseif isfield(param.spacing,'hdefault') && ~isempty(param.spacing.hdefault)
            space = param.spacing.hdefault;
        elseif isfield(param.spacing,'default') && ~isempty(param.spacing.default)
            space = param.spacing.default;
        else
            space = 4; %if all else fails
        end
    case 'uipanel'  %this catches uibuttongroups too
        if ~isempty(stylegroup) && isfield(param.(stylegroup),'afterspace') && param.(stylegroup).afterspace
            space = param.(getappdata(obj_handle,'StyleGroup')).afterspace;
        elseif isfield(param.spacing,'subcontainer') && isfield(param.spacing.subcontainer,'afterspace')
            space = param.spacing.subcontainer.afterspace;
        elseif isfield(param.spacing,'hdefault')
            space  = param.spacing.hdefault;
        elseif isfield(param.spacing,'default')
            space = param.spacing.default;
        else
            space = 0;
        end
    case 'axes'
        if isfield(param.(getappdata(obj_handle,'StyleGroup')),'afterspace')
            space = param.(getappdata(obj_handle,'StyleGroup')).afterspace;
        elseif isfield(param,'axes') && isfield(param.axes,'afterspace')
            space  = param.axes.afterspace;
        elseif isfield(param.spacing,'hdefault')
            space = param.spacing.hdefault;
        elseif isfield(param.spacing,'default')
            space = param.spacing.default;
        else
            space = 4;
        end
    case 'figure'
        error('LogiGUI:NoSubfigures','A figure cannot be an element in a layout, it can only be a parent of a layout.');
    otherwise
        error(['Type ', get(obj_handle,'Type'),' unknown']);
end

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

